/*  bsp_start()
 *
 * Modified for FADS823 by Josu Onandia <jonandia@fagorautomation.es>
 * Based on the FADS860
 *
 *  This routine starts the application.  It includes application,
 *  board, and monitor specific initialization and configuration.
 *  The generic CPU dependent initialization has been performed
 *  before this routine is invoked.
 *
 *  The MPC860 specific stuff was written by Jay Monkman (jmonkman@frasca.com)
 *
 *  COPYRIGHT (c) 1989-1999.
 *  On-Line Applications Research Corporation (OAR).
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.OARcorp.com/rtems/license.html.
 *
 *  $Id: bspstart.c,v 1.2 1999/11/17 17:51:09 joel Exp $
 */

#include <bsp.h>
#include <mpc823.h>
#include <rtems/libio.h>
 
#include <libcsupport.h>
 
#include <string.h>

#ifdef STACK_CHECKER_ON
#include <stackchk.h>
#endif

/*
 *  The original table from the application and our copy of it with
 *  some changes.
 */
extern rtems_configuration_table Configuration;

rtems_configuration_table  BSP_Configuration;

rtems_cpu_table Cpu_table;

char *rtems_progname;

/*
 *  Use the shared implementations of the following routines
 */
void bsp_postdriver_hook(void);
void bsp_libc_init( void *, unsigned32, int );

/*
 *  Function:   bsp_pretasking_hook
 *  Created:    95/03/10
 *
 *  Description:
 *      BSP pretasking hook.  Called just before drivers are initialized.
 *      Used to setup libc and install any BSP extensions.
 *
 *  NOTES:
 *      Must not use libc (to do io) from here, since drivers are
 *      not yet initialized.
 *
 */
 
void
bsp_pretasking_hook(void)
{
  extern int       _end;
  rtems_unsigned32  heap_start;

  heap_start = (rtems_unsigned32) &_end;

  /* Align the heap on a natural boundary (4 bytes?) */
  if (heap_start & (CPU_ALIGNMENT-1)) {
    heap_start = (heap_start + CPU_ALIGNMENT) & ~(CPU_ALIGNMENT-1);
  }
  /* set up a 2048K heap */
  bsp_libc_init((void *) heap_start, 2048 * 1024, 0);
  
#ifdef STACK_CHECKER_ON
  /*
   *  Initialize the stack bounds checker
   *  We can either turn it on here or from the app.
   */
  
  Stack_check_Initialize();
#endif
  
#ifdef RTEMS_DEBUG
  rtems_debug_enable( RTEMS_DEBUG_ALL_MASK );
#endif
}


void bsp_start(void)
{
  extern int _end;
  rtems_unsigned32  heap_start;
  rtems_unsigned32  ws_start;
  volatile rtems_unsigned32 *bcsr1, *bcsr4;

  /*
   *  Allocate the memory for the RTEMS Work Space.  This can come from
   *  a variety of places: hard coded address, malloc'ed from outside
   *  RTEMS world (e.g. simulator or primitive memory manager), or (as
   *  typically done by stock BSPs) by subtracting the required amount
   *  of work space from the last physical address on the CPU board.
   */

  /*
   *  Need to "allocate" the memory for the RTEMS Workspace and
   *  tell the RTEMS configuration where it is.  This memory is
   *  not malloc'ed.  It is just "pulled from the air".
   */

  heap_start = (rtems_unsigned32) &_end;
  if (heap_start & (CPU_ALIGNMENT-1))
    heap_start = (heap_start + CPU_ALIGNMENT) & ~(CPU_ALIGNMENT-1);


  ws_start = heap_start + (2048 * 1024); /* Skip memory used as program heap */
  if (ws_start & ((512 * 1024) - 1)) {  /* align to 512K boundary */
    ws_start = (ws_start + (512 * 1024)) & ~((512 * 1024) - 1);
  }

  BSP_Configuration.work_space_start = (void *)ws_start;
  BSP_Configuration.work_space_size = 512 * 1024; 

  /*
   *  initialize the CPU table for this BSP
   */

  Cpu_table.pretasking_hook = bsp_pretasking_hook;  /* init libc, etc. */
  Cpu_table.postdriver_hook = bsp_postdriver_hook;
  Cpu_table.interrupt_stack_size = 8 * 1024;

  /*
   * Josu: I have not checked the clicks/clocks numbers !
   */
  Cpu_table.clicks_per_usec = 1;  /* According to 4MHz external clock input source */
  Cpu_table.serial_per_sec = 10000000;
  Cpu_table.serial_external_clock = 1;
  Cpu_table.serial_xon_xoff = 0;
  Cpu_table.serial_cts_rts = 1;
  Cpu_table.serial_rate = 9600;
  Cpu_table.timer_average_overhead = 0;
  Cpu_table.timer_least_valid = 0;
  Cpu_table.clock_speed = 24000000;  /* System clock speed is 24MHz(4MHz * 6) */
  /*
   * Josu: It seems that this line is needed, although I'm not sure why!
   */
  Cpu_table.exceptions_in_RAM = 0;

  /* 
   * Call this in case we use TERMIOS for console I/O
   */
  m823_console_reserve_resources(&BSP_Configuration);

  /*
   * Motorola FADS823 evaluation board specific initialization
   */
  bcsr1 = (volatile rtems_unsigned32 *)((m823.memc.bank[1].br & 0xFFFFFFFE) + 4);
  bcsr4 = (volatile rtems_unsigned32 *)((m823.memc.bank[1].br & 0xFFFFFFFE) + 16);
  *bcsr1 &= ~0x01000000;  /* Enable RS232 port 1 which will be used as terminal console */
  *bcsr4 &= ~0x80000000;  /* Disable Ethernet port Diagnostic Loop-Back mode */
  *bcsr4 |= (0x40000000 | 0x20000000);  /* Disable Twisted Pair Full-Duplex & Signal Quality Error Test */
  *bcsr1 |= (0x00040000 | 0x10000000);  /* Disable RS232 port 2 and IRDA port because Ethernet is enabled */
  *bcsr1 &= ~0x20000000;  /* Enable Ethernet port */

  /*
   * Setup CPM interrupt level and enable CPM interrupt.
   * Note:
   *   These settings should be consistent with the actual
   *   CPM interrupt handler in "vectors.s".
   */
  m823.cpmic.cicr = 0x00E49F80; /* SCaP=SCC1, SCbP=SCC2, SCcP=SCC3, SCdP=SCC4, IRL=4, HP=SCC1, IEN=1 */
  m823.siu.simask |= M823_SIMASK_LVM4; /* Enable CPM interrupt request to CPU core */

  /*
   * SCC2 dedicated to Ethernet
   */
  m823.smc_reg[0].smcm=0;
  m823.pram.smc1.rbase=0;
  m823.pram.smc1.tbase=0;
  M823ExecuteRISC(M823_CR_OP_STOP_TX | M823_CR_CHAN_SMC1);

  /* 
   * Josu: Cache and MMU still not supported 
   * cache_mmu_init();
   */
}
