/*
#########################################################################
#                                                                       
#  FILENAME: INIT860.S                                                  
#                                                                       
#  DESCRIPTION: 
#                                                                       
#  This file contains the initialization software that is first executed. 
#  It initializes or defines the operation for the following areas:
#
#   - SIU (System Interface Unit)
#
#   - Memory Controller
#
#   - Defines Internal Memory Map (IMMR)
#
#   - Clock and Reset Circuitry
#
#   - External Bus Interface
#
#  NOTES:  
#
#  (1) This startup was assembled using the GNU Cross Compiler.
#
#  (2) We are not using simplified mnemonics in this example. When the user 
#      has determined that the assembler chosen supports a simplified 
#      mnemonic set, then the user is free to change mnemonics for clarity's
#      sake. Refer to PowerPC Microprocessor Family: The Programming 
#      Environments for 32-Bit Microprocessors book for details on these 
#      instructions. The simplified mnemonic set for assemblers that support 
#      them are listed in appendix F of this manual.
#
#  (3) R0 is used exclusively as a means for transferring immediate data. It 
#      will always have the value of 0.
#
#  (4) R4 is used exclusively to contain the IMMR base address.
#
#  (5) R1 is used exclusively as the stack pointer.
#
#
#  REFERENCES:
#
#  1. MPC860 PowerQUICC User's Manual
#  2. PowerPC Microprocessor Family: The Programming Environments for 32-Bit
#     Microprocessors
#
#  HISTORY:
#
#  jay 4/22/98    Initial Release
#  lhs 3/29/2000  Porting to RTEMS BSP for Motorola FADS860 evaluation board
#                 using GNU Cross Compiler
#  Josu Onandia 1/20/2001 <jonandia@fagorautomation.es>
#                 Ported to FADS823. I had several problems booting the board,
#                 so I decided to comment a lot of initializations out,
#                 and to put them in the init script of GDB. So this startup
#                 will only work if you boot through GDB+BDM and make all the inits
#                 in GDB, BEFORE loading the code into the board.
#                 See fads823.init
#  
#########################################################################
*/

/*
#----------------------------
# Gerneral Purpose Registers
#----------------------------
*/
#define r1  1
#define r0  0
#define r2  2
#define r3  3 
#define r4  4 
#define r5  5
#define r6  6
#define r7  7
#define r8  8
#define r9  9
#define r10 10
#define r11 11
#define r12 12
#define r13 13 
#define r14 14
#define r15 15
#define r16 16 
#define r17 17
#define r18 18
#define r19 19
#define r20 20
#define r21 21
#define r22 22
#define r23 23
#define r24 24
#define r25 25
#define r26 26
#define r27 27
#define r28 28
#define r29 29
#define r30 30
#define r31 31

/*
#----------------------------------------------------------------------------
# On-Chip Core Registers. These values represent the special purpose 
# registers used in this example. Refer to PowerPC Microprocessor Family: 
# The Programming Environments for 32-Bit Microprocessors book for the 
# complete set and the MPC860 Users Manual.
#----------------------------------------------------------------------------
*/
#define LR     8   /* Link register */
#define CTR    9   /* Counter register */
#define SRR0   26  /* Save/Restore Register 0 */
#define SRR1   27  /* Save/Restore Register 1 */
#define ICR    148 /* Interrupt Cause Register */
#define ICTRL  158 /* Instruction Control Register */
#define DER    149 /* Debug Enable Register */
#define DPDR   630 /* Development Port Data Register */
#define IMMR   638 /* Internal I/O base register */
#define IC_CST 560 /* ICache Control Status Register */
#define IC_ADR 561 /* ICache Address Register */
#define IC_DAT 562 /* ICache Data Register */
#define DC_CST 568 /* DCache Control Status Register */
#define DC_ADR 569 /* DCache Address Register */
#define DC_DAT 570 /* DCache Data Register */
#define MI_CTR 784 /* MMU Instrction Control Register */
#define MD_CTR 792 /* MMU Data Control Register */
#define MD_AP  794 /* MMU Data Access Permission Register */
#define MD_EPN 795 /* MMU Data Effective Page Number Register */
#define MD_TWC 797 /* MMU Data Tablewalk Control Register */
#define MD_RPN 798 /* MMU Instruction Real Page Number Register */
#define MI_AP  786 /* MMU Instruction Access Permission Register */
#define MI_EPN 787 /* MMU Instruction Effective Page Number Reg */
#define MI_TWC 789 /* MMU Instruction Tablewalk Control Register */
#define MI_RPN 790 /* MMU Instruction Real Page Number Register */
#define SPRG0  272 /* Special Purpose Register 0 */
#define SPRG1  273 /* Special Purpose Register 1 */
#define SPRG2  274 /* Special Purpose Register 2 */
#define SPRG3  275 /* Special Purpose Register 3 */

/*
#----------------------------------------------------------------------------
# Register Offset Definitions. All these values are offsets from the 
# Internal Memory Map Register (IMMR) base pointer. The base value is deter-
# mined by the ISB bits in the Hard Reset Configuration Word. This word is 
# read on the data bus or is a default of 0. See the reset section of the 
# user's manual. For a complete set of all the IMMR registers, refer to the 
# MPC860 Users Manual.
#----------------------------------------------------------------------------
*/
#define SIUMCR 0x000 /* SIU Module configuration */
#define SYPCR  0x004 /* SIU System Protection Control */
#define BR0    0x100 /* Base Register 0 */
#define OR0    0x104 /* Option Register 0 */
#define BR1    0x108 /* Base Register 1 */
#define OR1    0x10C /* Option Register 1 */
#define BR2    0x110 /* Base Register 2 */
#define OR2    0x114 /* Option Register 2 */
#define MCR    0x168 /* Memory Command */
#define MAMR   0x170 /* Machine A Mode Register */
#define MPTPR  0x17A /* Memory Periodic Timer Prescaler */
#define MDR    0x17C /* Memory Data */
#define TBSCR  0x200 /* Time Base Status and Control Register */
#define RTCSC  0x220 /* Real Timer Clock Status and Control */
#define PISCR  0x240 /* PIT Status and Control */
#define SCCR   0x280 /* System Clock Control Register */
#define PLPRCR 0x284 /* PLL, Low power & Reset Control Register */

/*
#----------------------------------------
# Instruction and Data Cache definition.
# Note: must load into bits 0-15
#----------------------------------------
*/
#define CacheUnlockAllCmd 0x0A00 /* Cache Unlock_All command */
#define CacheDisableCmd   0x0400 /* Cache Disable command */
#define CacheInvAllCmd    0x0C00 /* Cache Invalidate_All command */
#define CacheEnableCmd    0x0200 /* DCache_Enable Command */
#define CacheEnableBit    0x8000 /* Cache Enable bit in I/DC_CST */

   .section .entry,"ax",@progbits
   .globl start
   .globl text_addr
   .globl text_length
/*
#########################################################################
# Function: start
#
# Description: This is the main startup function
#
#  History:
#
#  2/23/98    jay      
#
#########################################################################			
*/
start:
  /*
   #-------------------------------------------------------------------------
   # When the MPC860 comes out of reset, the BCSR0 (Board Control and Status
   # Register) drives the data bus with 0x41600000. This becomes the Hard 
   # Reset Configuration Word (HRCW). The IP bit in the HRCW is set so the 
   # reset exception Vector location is 0x00000100. This method of driving 
   # the bus is one of two ways to program the HRCW. The other is to accept 
   # the internal MPC860 default value. The details are explained in the 
   # reset section in the users manual. The flash on my development board is 
   # 512K by 32 bits wide. CS0 is tied directly to the flash so CS0 is 
   # active for all addresses at reset. The vector table is re-located from 
   # flash to physical address 0x00000000 in DRAM once chip selects are 
   # defined later in this initialization code. 
   #-------------------------------------------------------------------------
   */
  /*
   #-------------------------------------------------------------------------
   # Re-program the Machine State Register (MSR). Re-program the MSR(IP) 
   # bit to ensure the vector interrupt table starts at 0x00000000 for all
   # exceptions that occur after the vector table is relocated to 0x00000000
   # further down in this code. The Machine Check Exception is enabled 
   # and The RI bit which makes exceptions recoverable.  
   #-------------------------------------------------------------------------
   */
   addis    r0,0,0         /* load r0 with 0. This register will always be 0 */
                           /* in this module. */

#   ori      r3,r0,0x1002   /* Set ME,RI flags */
#   mtmsr    r3
#   mtspr    SRR1,r3        /* Need SRR1 to match MSR */

  /*
   #-------------------------------------------------------------------------
   # Load the following physical address into the Link Register. Jump to that
   # address so that LR equals Program Counter (PC).
   #-------------------------------------------------------------------------
   */
   oris     r3,r0,sync_jump@h    /* load the address */
   ori      r3,r3,sync_jump@l
   mtspr    LR,r3 
   bclr     20,0                 /* jump unconditionally to address in Link */
                                 /* Register (LR) */

sync_jump:
  /*
   #-------------------------------------------------------------------------
   # Invalidate all TLB entries: This needs to be done because Translation
   # Lookaside Buffers values are undefined. The MMU comes out of reset
   # disabled but if the MMU were enabled and TLB's were not invalidated 
   # first, erratic behavior would result.
   #-------------------------------------------------------------------------
   */
   tlbia    /* This single command invalidates all TLB entries */

   isync    /* This ensures that no other instruction is fetched until tlbia is */
            /* completely finished */

  /*
   ##############################################
   # Call 8xx_init to finish chip initialization.
   ##############################################
   */
   bl mpc8xx_init    /* User specific init code */

  /*
   ###############################################
   # Call startup to finish system initialization.
   ###############################################
   */
   bl startup        /* LR = base_addr */
base_addr:      
/*
#-----------------------
# Parameters from linker
#-----------------------
*/
toc_pointer:    
   .long   s.got
bss_length:     
   .long   bss.size
bss_addr:       
   .long   bss.start
text_addr:
   .long   text.start
text_length:
   .long   text.size

/*
#########################################################################
# Function: startup
#
# Description: Initialize C run-time environment and jump to C routine
#
#  History:
#
#  3/29/2000   lhs
#
#########################################################################			
*/
startup:
   mflr r1 /* Get start address */

   bl   bss_clr /* Clear .bss section */

   lwz  r2,toc_pointer-base_addr(r1) /* Set toc */

   addi r1,r1,-56-4 /* Set up stack pointer = beginning of text section - 56 */

   xor  r3,r3,r3  /* Clear argc */
   xor  r4,r4,r4  /* Clear argv */
   xor  r5,r5,r5  /* Clear envp */
   .extern boot_card
   b    boot_card /* Jump to the first C routine */

/*
#########################
# bss_clr - zero out bss
#########################
*/
bss_clr:
   lwz     r4, bss_addr-base_addr(r1)   /* Start of bss */
   lwz     r5, bss_length-base_addr(r1) /* Length of bss */

   rlwinm. r5,r5,30,0x3FFFFFFF          /* form length/4 */
   beqlr                                /* no bss */
   mtctr   r5                           /* set ctr reg */
   xor     r6,r6,r6                     /* r6 = 0 */
clear_loop:
   stswi   r6,r4,0x4                    /* store r6 */
   addi    r4,r4,0x4                    /* update r2 */

   bdnz    clear_loop                   /* dec counter and loop */
   blr                                  /* return */

/*
#########################################################################
# Function: 8xx_init
#
# Description: This function initializes the rest of the 8xx with the 
#              exception of the CPM.
#
#  History:
#
#  4/28/98    jay      
#
#########################################################################			
*/
mpc8xx_init:

   mfspr    r31,LR        /* Save the Link Register value */

  /*
   #-----------------------------------------------------------------------
   # Program the Data Cache Control and Status Register
   #
   # - DEN (Data Cache Enable Status Bit) = 0 = Disabled
   #
   # - DFWT (Data Cache Force Writethrough) = 0 = Data cache is determined
   #   by MMU.
   #
   # - LES (Little Endian Swap) = 0 = Address of the Data and the Instruction
   #   Caches is the unchanged address from the core. No byte swapping is 
   #   done on the data and instruction caches external accesses.
   #
   # - CMD (Data Cache Commands) = 0100 = Data Cache Disable
   #
   # - All other bits are status related and are read only
   #-----------------------------------------------------------------------
   */
   oris     r3,r0,CacheDisableCmd     /* DCache Disable command */
   ori      r3,r3,0
   mtspr    DC_CST,r3

  /*
   #----------------------------------------------
   # Load the IMMR register with the base address
   #----------------------------------------------
   */
   oris     r4,r0,0xFF00       /* IMMR base addr = 0xFF000000 */
   mtspr    IMMR,r4

  /*
   #-----------------------------------------------------------------------
   # Program the System Protection Control Register with the following 
   # values:
   #
   # - SWTC (Software Watchdog Timer Count) = 0xFFFF = programmed to max
   #   count value.
   #
   # - BMT (Bus Monitor Timing) = 0xFF = Set to max timeout period.
   #
   # - BME (Bus Monitor Enable) = 1 = enabled.
   #
   # - SWF (Software Watchdog Freeze) = 1 = The software timer stops when
   #   freeze is asserted.
   #
   # - SWE (Software Watchdog Enabled) = 0 = disabled for now. User will 
   #   have to enable this in a non-debug application.
   #
   # - SWRI (Software Watchdog Reset/Interrupt Select) = 0 = Software 
   #   watchdog timeout causes a nonmaskable interrupt to the PowerPC core.
   #
   # - SWP (Software Watchdog Prescale) = 0 = It's not prescaled.
   #-----------------------------------------------------------------------
   */
#   oris     r3,r0,0xFFFF
#   ori      r3,r3,0xFF88   /* SYPCR = 0xFFFFFF88 */
#   stw      r3,SYPCR(r4)    

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~
   # Initialize Chip Selects
   #~~~~~~~~~~~~~~~~~~~~~~~~~
   */
#   bl       init_memc

  /*
   #~~~~~~~~~~~~~~~~~~~~
   # Initialize the SIU
   #~~~~~~~~~~~~~~~~~~~~
   */
   bl       init_siu

  /*
   #-------------------------------------------------------------------------
   # Program the Instruction Support Control Register (ICTRL). This is a 
   # debug related register and it's description can be found in the back
   # of the Development Support section of the user's manual. Basically the
   # only thing being programmed is ISCT_SER and it's value is 110b. This 
   # means that the core is not serialized (Normal mode) and show cycle will
   # be performed for all indirect changes in the program flow if show cycles
   # are enabled. In this application, it is disabled.
   #-------------------------------------------------------------------------
   */
#   oris     r3,r0,0x0000    
#   ori      r3,r3,0x0006
#   mtspr    ICTRL,r3

  /*
   #-------------------------------------------------------------------------
   # Program the Debug Enable Register (DER). This register provides the user
   # with the reason for entering into the debug mode. We want all conditions
   # to end up as an exception. We don't want to enter into debug mode for 
   # any condition. See the back of of the Development Support section of the 
   # MPC860 User Manual for a description of this register.
   #-------------------------------------------------------------------------
   */
#   oris     r3,r0,0x0000    
#   mtspr    DER,r3         /* write 0 */

  /*
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   # Clocks, PLL and Power Programming
   #~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   */
   bl       init_clks

  /*
   #~~~~~~~~~~~~~~~~~~~~
   # end of mpc8xx_init
   #~~~~~~~~~~~~~~~~~~~~
   */
   mtspr    LR,r31      /* restore original Link Register value */

   bclr     20,0        /* jump unconditionally to effective address in Link */
                        /* register */

/*
#########################################################################
# Function: init_memc
#
# Description: This function initializes chip selects for memory and I/O
#              devices. Chip select 0 is assigned to the flash device.
#              Chip select 1 is assigned to the BCSR (Board Control
#              and Status Registers).
#
#  History:
#
#  4/28/98    jay      
#
#########################################################################			
*/
init_memc:
  /*
   #-------------------------------------------------------------------------
   # Finish the Memory Controller by initializing the Base Registers and the
   # Options Registers for the active memory banks
   #-------------------------------------------------------------------------
   */
  /*
   #-------------------------------------------------------------------------
   # Base Register 0 (BR0): Bank 0 is assigned to the 2Mbyte (512K X 32) 
   #                        flash that resides on the MPC860 ADS board. The 
   #                        particulars are defined here as well as 
   #                        the UPM table.
   #
   # BA (Base Address) = 0x0280+0b for a total of 17 bits. 17 bit base addr
   #                     represents a minumum 32K block size.
   #
   # AT (Address Type) = 000b = no restrictions to bank
   # PS (Port Size) = 00b = 32 bit port size
   # PARE (Parity Enable) =  0 = parity checking disabled
   # WP (Write Protect) = 0 = both read and write accesses are allowed
   # MS (Machine Select) = 00 = General Purpose Chip Select machine Selected
   # V (Valid Bit) =  1 = Valid bit set
   #-------------------------------------------------------------------------
   */
   oris     r3,r0,0x0280   /* R3 holds the value temporarily */
   ori      r3,r3,0x0001

  /*
   #-------------------------------------------------------------------------
   # Option Register 0 (OR0): further flash definitions
   #
   # AM (Address Mask) = 0xFFE0+0b = We've masked the upper 11 bits which 
   #                                 which defines a 2 Mbyte memory block.
   #
   # ATM (Address Type Mask) = 000 = ignore addr type codes as a part of the
   #                                 addr comparison.
   #
   # CSNT/SAM (Chip Select Negation Time) = 1 = CS/|WE/ are negated a quarter
   #                                            of a clock early.
   #
   # ACS/G5LA,G5LS (Address To Chip-Select Setup) = 10 = CS/ is a output a 
   #                                                     quarter of a clk 
   #                                                     later than the addr
   #                                                     lines.
   #
   # BI/ (Burst Inhibit) = 1 = Drive BI/ asserted. This bank does not support
   #                           burst accesses.
   #
   # SCY (Cycle Length In Clocks) = 0101 = Add a 5 clock cycle wait state
   #
   # SETA (External Transfer Acknowledge) = 0 = TA/ is generated internally 
   #                                            by the memory controller 
   #                                            unless asserted earlier 
   #                                            externally.
   #
   # TRLX (Timing Relaxed) = 1 = Relaxed timing is generated by the GPCM.
   #
   # EHTR (Extended Hold Time On Read Accesses) = 0 = Normal timing is 
   #                                                  generated by the memory
   #                                                  controller
   #-------------------------------------------------------------------------
   */
   oris     r5,r0,0xFFE0     /* R5 holds the value temporarily */
   ori      r5,r5,0x0954   

  /*
   #------------------------------------------------------------------------
   # It's important to note the order in which OR0 and BR0 are programmed.
   # When coming out of reset and CS0 is the global chip select, OR0 MUST be 
   # programmed AFTER BR0. In all other cases BRx would be programmed after
   # ORx.
   #------------------------------------------------------------------------
   */
  /*
   #------------------
   # Write the values
   #------------------
   */
   stw        r3,BR0(r4)
   stw        r5,OR0(r4)

  /*
   #-------------------------------------------------------------------------
   # Program Base Register 1 (BR1): This section further defines the inter-
   #                                face with the Board Control Status 
   #                                Registers (BCSR).There are 4 Programmable
   #                                Logic chips on the ADS board.
   #
   # BA (Base Address) = 0x0210+0b for a total of 17 bits. 17 bit base addr
   #                     represents a minumum 32K block size.
   #
   # AT (Address Type) = 000b = no restrictions to bank
   # PS (Port Size) = 00b = 32 bit port size
   # PARE (Parity Enable) =  0 = parity checking disabled
   # WP (Write Protect) = 0 = both read and write accesses are allowed
   # MS (Machine Select) = 00 = General Purpose Chip Select machine Selected
   # V (Valid Bit) =  1 = Valid bit set
   #-------------------------------------------------------------------------
   */
   oris     r3,r0,0x0210     /* R3 holds the value temporarily */
   ori      r3,r3,0x0001

  /*
   #-------------------------------------------------------------------------
   # Program Option Register 1 (OR1): further BCSR definitions
   #
   # AM (Address Mask) = 0xFFFF+1b = We've masked the upper 17 bits which 
   #                                 represents a 128K block for the 4 PLDs
   #                                 on the target. 
   #
   # ATM (Address Type Mask) = 000 = ignore addr type codes as a part of the
   #                                 addr comparison.
   #
   # CSNT/SAM (Chip Select Negation Time) = 0 = CS/|WE/ are negated normally.
   #
   # ACS/G5LA,G5LS (Address To Chip-Select Setup) = 00 = CS/ is output at the
   #                                                     same time as the 
   #                                                     addr lines.
   #
   # BI/ (Burst Inhibit) = 1 = Drive BI/ asserted. This bank does not support
   #                           burst accesses.
   #
   # SCY (Cycle Length In Clocks) = 0001 = Add a 1 clock cycle wait state
   #
   # SETA (External Transfer Acknowledge) = 0 = TA/ is generated internally 
   #                                            by the memory controller 
   #                                            unless asserted earlier 
   #                                            externally.
   #
   # TRLX (Timing Relaxed) = 0 = Normal timing is generated by the GPCM.
   #
   # EHTR (Extended Hold Time On Read Accesses) = 0 = Normal timing is 
   #                                                  generated by the memory
   #                                                  controller
   #-------------------------------------------------------------------------
   */
   oris     r5,r0,0xFFFF     /* R5 holds the value temporarily */
   ori      r5,r5,0x8110

  /*
   #------------------
   # Write the values
   #------------------
   */
   stw        r5,OR1(r4)
   stw        r3,BR1(r4)

  /*
   #-----------------------
   # return from init_memc
   #-----------------------
   */
   bclr  20,0    /* jump unconditionally to effective address in Link */
                 /* register */

/*
#########################################################################
# Function: init_siu
#
# Description: Initializes the System Interface Unit
#
#  History:
#
#  4/28/98    jay      
#
#########################################################################			
*/
init_siu:
  /*
   #-------------------------------------------------------------------------
   # Program the SIU Module Configuration Register (SIUMCR) register as 
   # follows:
   #
   # - EARB (External Arbitration) = 0 = Internal Arbitration is used.
   #
   # - EARP (External Arbitration Request Priority) = 000 = Ignored in
   #   particular configuration.
   #
   # - DS HW (Data Show Cycles) = 0 = Disable show cycles for all internal
   #   data cycles.
   #
   # - DBGC (Debug Pins Configuration) = 00 = see bit configuration 
   #   description in manual.
   #
   # - DBPC (Debug Port Pins Configuration) = 00 = see bit configuration
   #   description in manual.
   #
   # - FRC (FRZ/IRQ6 pin Configuration) = 1 = configure to be IRQ6.
   #
   # - DLK (Debug Register Lock) = 1 = Bits 8:15 are locked.
   #
   # - OPAR (Odd Parity) = 0 = Even selected. This bit ignored for this 
   #   application.
   #
   # - PNCS (Parity Enable For Non-memory Controller Regions) = 0 = parity 
   #   disabled
   #
   # - DPC (Data Parity Pins Configuration) = 1 = Configures DP(0:3)/IRQ(3:6)
   #   functions as DP(0:3).
   #
   # - MPRE (Multi-processors Reservation Enable) = 0 = Inter-processor
   #   communication protocol is disabled.
   #
   # - MLRC (Mult-Level Reservation Control) = 01 = RETRY/|KR/|IRQ4/|SPKROUT
   #   is tri-stated.
   #
   # - AEME (Asyncronous External Master Enable) = 1 = any assertion of 
   #   the AS/ pin is interpreted by the memory controller as an external
   #   asynchronous master initiating a transfer.
   #
   # - SEME (Synchronous External Master Enable) = 0 = Ignore the TS/ pin
   #
   # - BSC (Byte Select Configuration) = 0 = see bit configuration 
   #   description in user's manual.
   #
   # - GB5E (GPL_B(5) Enable) = 1 = GPL_B(5)/ of the memory controller 
   #   functionality is active.
   #
   # - B2DD (Bank 2 Double Drive) = 0 = CS2/ line is not reflected on 
   #   GPL_A2/|GPL_B2/.
   #
   # - B3DD (Bank 3 Double Drive) = 0 = CS3/ line is not reflected on 
   #   GPL_A3/|GPL_B3/.
   #
   #-------------------------------------------------------------------------
   */
#   lwz        r3,SIUMCR(r4) /*  SIUMCR = The old value bitwize */
                            /*           ored with 0x00032640  */
#   addis      r5,0,0x0003
#   ori        r5,r5,0x2640
#   or         r3,r3,r5
#   stw        r3,SIUMCR(r4)

  /*
   #-------------------------------------------------------------------------
   # Program the Periodic Interrupt Status and Control Register (PISCR) as 
   # follows:
   #
   # - PS (Periodic Interrupt Request Level) = 1 = negate bit value by 
   #   writing a one to it.
   #
   # - PIE (Periodic Interrupt Enable) = 0 = Don't generate an interrupt when
   #   PS bit is asserted.
   #
   # - PITF (Periodic Interrupt Timer Freeze) = 1 = The periodic interrupt
   #   timer stops while freeze is asserted.
   #
   # - PTE (Periodic Timer Enable) = 0 = Disable
   #-------------------------------------------------------------------------
   */
   addi        r3,0,0x0082     /*  PISCR = 0x0082 (16bit register) */
   sth         r3,PISCR(r4)

  /*
   #-----------------------
   # return from init_siu
   #-----------------------
   */
   bclr  20,0           /* jump unconditionally to effective address in Link */
                        /* register */

/*
#########################################################################
# Function: init_clks
#
# Description: 
#
# This function programs the System Clock Control Register (SCCR) and
# the PLL,Low Power, and Reset Control Register (PLPRCR). Basically, the
# 4 Mhz oscillator clk is selected and multiplied up to 24Mhz. This 
# frequency is also the CLKOUT frequency.
#
#  History:
#
#  4/28/98    jay      
#
#########################################################################			
*/
init_clks:
  /*
   #-------------------------------------------------------------------------
   # The System Clock Control Register (SCCR) is not actually programmed 
   # here. We are mostly relying on the reset default values. The rest of
   # the parameters in this register are read from pins or the Hard Reset
   # Configuration Word. These pins that are read are MODCK1 and MODCK2. 
   # These two pins are controlled, thru a PAL by a dipswitch on the 
   # MPC860ADS Board. Here are possible values for our target:
   #
   #   MODCK1&MODCK2 = 00 = Normal Operation, PLL enabled. Main Timing Ref.
   #                        is freq(Xtal input) = 32 KHz
   # 
   #   MODCK1&MODCK2 = 11 = Normal Operation, PLL enabled. Main Timing Ref.
   #                        is freq(EXTCLK) = 4 MHz. This is the option we've
   #                        selected.
   #
   # 0x01800000 has been chosen as the default value. Here are descriptions 
   # of each bit and the values we've chosen:
   #
   # - COM[0:1] (Clock Output Mode) = 00 = Clock output enabled full-strength
   #                                      output buffer.
   #
   # - TBS (Timebase Source) = 0 = Source is oscclk divided by (4 or 16)
   #
   # - RTDIV (RTC clock divide) = value of MODCK1&MODCK2 pins = 11 =
   #   Divide by 512 for Real Time Clock module. This frequency is 4Mhz/512
   #   which equals 7812 cycles/sec.
   #
   # - RTSEL (Real Time Clock Circuit Input Source Select) = value of 
   #   MODCK1 pin = 1 in this application. EXTCLK is selected as input to RTC
   #   and PIT.
   #
   # - CRQEN (CPM Request Enable) = 0 = The system remains in the lower freq.
   #   (defined by the DFNL bits), even if the CPM is active
   #
   # - PRQEN (Power Management Request Enable) = 0 = The system remains in 
   #   the lower frequency (defined by the DFNL bits) even if there is a  
   #   pending interrupt from the interrupt controller or the power 
   #   management bit in the MSR is cleared (normal operational mode).
   #
   # - EBDF[0:1] - External Bus Division Factor = 00 = GCLKx_50 and CLKOUT
   #   have a 1:1 relationship to GCLKx and GCLKxC.
   #
   # - DFSYNC[0:1] - Division Factor for the SYNCCLK. This clock is used by
   #   the CPM for communications related activities. It's value is 00 which
   #   gives a 1:1 relationship between VCOCLK and SYNCCLK.
   #
   # - DFBRG[0:1] - Division Factor for the BRGCLK = 00 = The relationship
   #   between VCOCLK and BRGCLK is 1:1.
   #
   # - DFNL[0:2] - Divsion Factor Lowest Frequency = 000 = In the case of 
   #   low power mode. The frequency relationship between VCOCLK and 
   #   GCLKx,GCLKxC,GCLKx_50, and CLKOUT is 1:1/2.
   #
   # - DFNH[0:2] - Divsion Factor Highest Frequency = 000 = In the case of 
   #   normal mode. The frequency relationship between VCOCLK and 
   #   GCLKx,GCLKxC,GCLKx_50, and CLKOUT is 1:1.
   #
   #-------------------------------------------------------------------------
   */
  /*
   #-------------------------------------------------------------------------
   # We're reading this bit to determine if the crystal or the 4Mhz ocillator
   # is being used because MODCLK1 controls this selection and it controls
   # the value of this bit by the same logical value.
   #
   #     MODCLK1:
   #
   #        0 - Main clock oscillator from EXTCLK (4Mhz designed in)
   #        1 - Crystal oscillator
   #
   # In this example, we've chosen the 4 Mhz crystal oscillator.
   #-------------------------------------------------------------------------
   */
   lwz	      r3,SCCR(r4)	  /* reading system clock register */
   andis.     r3,r3,0x0080        /* check for bit 8, RTSEL */
   bc         12,2,Crystal        /* branch if equal */

  /*
   #---------------------------------
   # Now program the PLPRCR register
   #---------------------------------
   */
  /*
   #-------------------------------------------------------------------------
   # Program the PLPRCR (PLL, Low Power, and Reset Control) register with the
   # following values
   #
   # - MF (Multiplication Factor of SPLL) = 0x005 = Set the ADS board to 
   #   24MHZ from 4MHZ clock at EXTCLK. MF value + 1 = 6. 6*4Mhz = 24 Mhz.
   #
   # - SPLSS (System PLL Lock Status Sticky Bit) This is a status bit.
   #
   # - TEXPS (Timer Expired Status Bit) = 1 = Status bit is cleared with 1
   #
   # - TMIST (Timers Interrupt Status Bit) = This is a status bit.
   #
   # - CSRC (Clock Source Bit) = 0 = General system clock is determined by 
   #   the DFNH value.
   #
   # - LPM[0:1] (Low Power Modes) = 00 = See LPM bit description in manual.
   #
   # - CSR (Checkstop Reset Enable) = 0 = Disabled
   #
   # - LOLRE (Loss of Lock Reset Enable) = 0 = Loss of lock does not cause
   #   reset.
   #
   # - FIOPD (Force I/O Pull Down) = 0 = The address and data pins are not
   #   driven by an internal pull-down device during at sleep or deep sleep
   #   low power modes.
   #
   # Note: bits 31-16 didn't need to be programmed and remained at their 
   #       default values.
   #-------------------------------------------------------------------------
   */
   addis        r3,0,0x0050     /* multiply factor from 4MHZ at EXTCLK */

   b	SetPLL

Crystal:
  /*
   #-------------------------------------------------------------------------
   # Program the PLPRCR (PLL, Low Power, and Reset Control) register with the
   # following values for the case of a crystal selected as the input 
   # frequency.
   #
   # - MF (Multiplication Factor of SPLL) = 0x2DC = Set the ADS board to 
   #   24MHZ from 32.768Khz xtal input. MF value + 1 = 733. 733*32.768Khz = 
   #   24.019 Mhz.
   #
   # - SPLSS (System PLL Lock Status Sticky Bit) This is a status bit.
   #
   # - TEXPS (Timer Expired Status Bit) = 1 = Status bit is cleared with 1
   #
   # - TMIST (Timers Interrupt Status Bit) = This is a status bit.
   #
   # - CSRC (Clock Source Bit) = 0 = General system clock is determined by 
   #   the DFNH value.
   #
   # - LPM[0:1] (Low Power Modes) = 00 = See LPM bit description in manual.
   #
   # - CSR (Checkstop Reset Enable) = 0 = Disabled
   #
   # - LOLRE (Loss of Lock Reset Enable) = 0 = Loss of lock does not cause
   #   reset.
   #
   # - FIOPD (Force I/O Pull Down) = 0 = The address and data pins are not
   #   driven by an internal pull-down device during at sleep or deep sleep
   #   low power modes.
   #
   # Note: bits 31-16 didn't need to be programmed and remained at their 
   #       default values.
   #-------------------------------------------------------------------------
   */
   addis    r3,0,0x2DC0         /* multiply factor from 32.768KHZ */

  /*
   #-------------------------
   # Perform the actual write 
   #-------------------------
   */
SetPLL:

#   stw      r3,PLPRCR(r4)   /* set the PLL register */

  /*
   #-----------------------
   # return from init_clks
   #-----------------------
   */
   bclr  20,0           /* jump unconditionally to effective address in Link */
                        /* register */

/*
############################################################################
# This is debug code that I used when debugging out of flash. It toggles 
# Port A, bit 0. It allowed me, with an O'scope to verify code was executing 
# at various locations in this module. It can be turned into a subroutine. 
# I cut and pasted it where I wanted it. On the ADS evaluation board Port A 
# bit 0 can be monitored on P10-A18.
############################################################################

#---------------------------------------------------------------------------
#   addis    r0,0,0        # load r0 with 0. This value will always be 0 in
#                          # this module.
#
#   oris     r4,r0,0xFF00       # IMMR = 0x0FF00000:power-on-reset value
#   mtspr    IMMR,r4
#
#   addi     r28,0,0x8000
#   sth      r28,0x950(r4)
#loop_here:
#   addi     r28,0,0x8000
#   sth      r28,0x956(r4)
#   addi     r28,0,0x0000
#   sth      r28,0x956(r4)
#   b        loop_here
#---------------------------------------------------------------------------
*/
