/*
 * RTEMS/TCPIP driver for MPC860 SCC1 Ethernet
 *
 * Modified for MPC823 by Josu Onandia <jonandia@fagorautomation.es>
 * Only supports SCC2, because it's the only Ethernet port in the MPC823
 *
 * Modified for MPC860 by Jay Monkman (jmonkman@frasca.com)
 *
 * This supports ethernet on either SCC1 or the FEC of the MPC860T.
 *  Right now, we only do 10 Mbps, even with the FEC. The function
 *  rtems_m860_enet_driver_attach determines which one to use. Currently,
 *  only one may be used at a time.
 *
 * W. Eric Norum
 * Saskatchewan Accelerator Laboratory
 * University of Saskatchewan
 * Saskatoon, Saskatchewan, CANADA
 * eric@skatter.usask.ca
 *
 *  $Id: network.c,v 1.2 1999/02/18 15:09:59 joel Exp $
 */
#include <bsp.h>
#include <mpc823.h>
#include <stdio.h>
#include <rtems/error.h>
#include <rtems/rtems_bsdnet.h>

#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/sockio.h>

#include <net/if.h>

#include <netinet/in.h>
#include <netinet/if_ether.h>

/*
 * Number of interfaces supported by this driver
 */
#define NIFACES 1

/*
 * Default number of buffer descriptors set aside for this driver.
 * The number of transmit buffer descriptors has to be quite large
 * since a single frame often uses four or more buffer descriptors.
 */
#define RX_BUF_COUNT     32
#define TX_BUF_COUNT     8
#define TX_BD_PER_BUF    4

/*
 * RTEMS event used by interrupt handler to signal daemons.
 * This must *not* be the same event used by the TCP/IP task synchronization.
 */
#define INTERRUPT_EVENT RTEMS_EVENT_1

/*
 * RTEMS event used to start transmit daemon.
 * This must not be the same as INTERRUPT_EVENT.
 */
#define START_TRANSMIT_EVENT    RTEMS_EVENT_2

/*
 * Receive buffer size -- Allow for a full ethernet packet plus a pointer
 */
#define RBUF_SIZE       1520

#if (MCLBYTES < RBUF_SIZE)
# error "Driver must have MCLBYTES > RBUF_SIZE"
#endif

/*
 * Per-device data
 */
struct m823_enet_struct {
        struct arpcom           arpcom;
        struct mbuf             **rxMbuf;
        struct mbuf             **txMbuf;
        int                     acceptBroadcast;
        int                     rxBdCount;
        int                     txBdCount;
        int                     txBdHead;
        int                     txBdTail;
        int                     txBdActiveCount;
        m823BufferDescriptor_t  *rxBdBase;
        m823BufferDescriptor_t  *txBdBase;
        rtems_id                rxDaemonTid;
        rtems_id                txDaemonTid;

        /*
         * Statistics
         */
        unsigned long   rxInterrupts;
        unsigned long   rxNotFirst;
        unsigned long   rxNotLast;
        unsigned long   rxGiant;
        unsigned long   rxNonOctet;
        unsigned long   rxRunt;
        unsigned long   rxBadCRC;
        unsigned long   rxOverrun;
        unsigned long   rxCollision;

        unsigned long   txInterrupts;
        unsigned long   txDeferred;
        unsigned long   txHeartbeat;
        unsigned long   txLateCollision;
        unsigned long   txRetryLimit;
        unsigned long   txUnderrun;
        unsigned long   txLostCarrier;
        unsigned long   txRawWait;
};
static struct m823_enet_struct enet_driver[NIFACES];

/*
 * SCC2 interrupt handler
 */
static rtems_isr
m823_scc2_interrupt_handler (rtems_vector_number v)
{
        /*
         * Frame received?
         */
        if ((m823.scc_reg.sccm & 0x8) && (m823.scc_reg.scce & 0x8)) {
                m823.scc_reg.scce = 0x8;
                /* I don't think the next line is needed. It was in 
                 * the 68360 stuff, though.
                 *   m823.scc2.sccm &= ~0x8; 
                 */
                enet_driver[0].rxInterrupts++;
                rtems_event_send (enet_driver[0].rxDaemonTid, INTERRUPT_EVENT);
        }

        /*
         * Buffer transmitted or transmitter error?
         */
        if ((m823.scc_reg.sccm & 0x12) && (m823.scc_reg.scce & 0x12)) {
                m823.scc_reg.scce = 0x12;
                /* I don't think the next line is needed. It was in 
                 * the 68360 stuff, though.
                 *   m823.scc2.sccm &= ~0x12; 
                 */
                enet_driver[0].txInterrupts++;
                rtems_event_send (enet_driver[0].txDaemonTid, INTERRUPT_EVENT);
        }
        m823.cpmic.cisr = 1UL << 29;  /* Clear SCC2 interrupt-in-service bit */
}

/*
 * Initialize the ethernet hardware
 */
static void
m823_scc_initialize_hardware (struct m823_enet_struct *sc)
{
  int i;
  unsigned char *hwaddr;
  rtems_status_code status;
  rtems_isr_entry old_handler;
    /* Get the base address of BCSR */
    unsigned long *bcsr = (unsigned long *) (m823.memc.bank[1].br & 0xfffffffe);
  
    /*
     * Configure Ethernet Transmit/Receive Clocks from MC68160
     * See FADS Board Pins P8-13,15.  ETHTCK/CLK3=PA5, ETHRCK/CLK2=PA6.
     *
     * Configure Port A Pins for SCC2 ETHRX/RXD2=PA13 and ETHTX/TXD2=PA12
     */
	m823.pio.papar |= (0x060c);	/* (0x030c); */
	m823.pio.padir &= ~(0x060c);	/* (0x030c); */
	m823.pio.paodr &= ~(0x0008);
  
    /*
     * Configure Ethernet  E_CLSN/CTS2*=PC9, E_RENA/CD2*=PC8 Signals
     */
	
	m823.pio.pcpar &= ~(0x00c0);
	m823.pio.pcdir &= ~(0x00c0);
	m823.pio.pcso |= (0x00c0);
	
    /*
	* Initialize SICR -- see section 13.12.5.3 (p. 478)
	* Settings:
	*     SCC1:  NMSI,  Transmit Clock=CLK3, Receive Clock=CLK2
	*/
	
	m823.si.sicr &= ~(0xff00);          /* Clear previous scc2 setting */
	m823.si.sicr |= 0x2e00;             /* Setup scc2 */
  
  /*
   * Initialize SDMA configuration register
   */
  m823.siu.sdcr = 1;
    /*
     * Setup SCC2 Ethernet Parameter RAM
     */

    m823.pram.scc2.rfcr = 0x18;                                                /* Normal Operation and Mot byte ordering */
    m823.pram.scc2.tfcr = 0x18;                                                 /* Mot byte ordering, Normal access */

    m823.pram.scc2.mrblr = RBUF_SIZE;

  
  /*
   * Allocate mbuf pointers
   */
  sc->rxMbuf = malloc (sc->rxBdCount * sizeof *sc->rxMbuf, 
		       M_MBUF, M_NOWAIT);
  sc->txMbuf = malloc (sc->txBdCount * sizeof *sc->txMbuf, 
		       M_MBUF, M_NOWAIT);
  if (!sc->rxMbuf || !sc->txMbuf)
    rtems_panic ("No memory for mbuf pointers");
  
  /*
   * Set receiver and transmitter buffer descriptor bases
   */
  sc->rxBdBase = M823AllocateBufferDescriptors(sc->rxBdCount);
  sc->txBdBase = M823AllocateBufferDescriptors(sc->txBdCount);
  m823.pram.scc2.rbase = (char *)sc->rxBdBase - (char *)&m823;
  m823.pram.scc2.tbase = (char *)sc->txBdBase - (char *)&m823;
  

  /*
   * Send "Init parameters" command
   */
  M823ExecuteRISC (M823_CR_OP_INIT_RX_TX | M823_CR_CHAN_SCC2);

    /*
     * Ethernet Specific Parameter RAM
     *     see table 13-16, pg. 660,
     *     pg. 681 (example with suggested settings)
     */

    m823.pram.scc2_spi.e.c_pres = ~(0x0);                             /* Preset CRC */
    m823.pram.scc2_spi.e.c_mask = 0xdebb20e3;                         /* Constant Mask for CRC */
    m823.pram.scc2_spi.e.crcec = 0x0;                                 /* Error Counter CRC (unused) */
    m823.pram.scc2_spi.e.alec = 0x0;                                  /* Alignment Error Counter (unused) */
    m823.pram.scc2_spi.e.disfc = 0x0;                                 /* Discard Frame Counter (unused) */
    m823.pram.scc2_spi.e.pads = 0x8888;                               /* Short Frame PAD Characters */

    m823.pram.scc2_spi.e.ret_lim = 0xf;                               /* Retry Limit Threshold */
    m823.pram.scc2_spi.e.mflr = 1518;                                 /* MAX Frame Length Register */
    m823.pram.scc2_spi.e.minflr = 64;                                 /* MIN Frame Length Register */

    m823.pram.scc2_spi.e.maxd1 = RBUF_SIZE;                                /* MAX DMA1 Length Register */
    m823.pram.scc2_spi.e.maxd2 = RBUF_SIZE;                                /* MAX DMA2 Length Register */

    m823.pram.scc2_spi.e.gaddr1 = 0x0;                                /* Group Address Filter 1 (unused) */
    m823.pram.scc2_spi.e.gaddr2 = 0x0;                                /* Group Address Filter 2 (unused) */
    m823.pram.scc2_spi.e.gaddr3 = 0x0;                                /* Group Address Filter 3 (unused) */
    m823.pram.scc2_spi.e.gaddr4 = 0x0;                                /* Group Address Filter 4 (unused) */

  /*
   * Set our physical address
   */
  hwaddr = sc->arpcom.ac_enaddr;
  
  m823.pram.scc2_spi.e.paddr_l = (hwaddr[5] << 8) | hwaddr[4];
  m823.pram.scc2_spi.e.paddr_m = (hwaddr[3] << 8) | hwaddr[2];
  m823.pram.scc2_spi.e.paddr_h = (hwaddr[1] << 8) | hwaddr[0];

    m823.pram.scc2_spi.e.p_per = 0x0;                                 /* Persistence (unused) */
    m823.pram.scc2_spi.e.iaddr1 = 0x0;      /* Individual Address Filter 1 (unused) */
    m823.pram.scc2_spi.e.iaddr2 = 0x0;      /* Individual Address Filter 2 (unused) */
    m823.pram.scc2_spi.e.iaddr3 = 0x0;      /* Individual Address Filter 3 (unused) */
    m823.pram.scc2_spi.e.iaddr4 = 0x0;                                /* Individual Address Filter 4 (unused) */
    m823.pram.scc2_spi.e.taddr_h = 0x0;                               /* Tmp Address (MSB) (unused) */
    m823.pram.scc2_spi.e.taddr_m = 0x0;                               /* Tmp Address (unused) */
    m823.pram.scc2_spi.e.taddr_l = 0x0;                               /* Tmp Address (LSB) (unused) */

    /*
     * Clear Events in SCCE -- Clear bits by writing 1's
     */

    m823.scc_reg.scce = ~(0x0);

	
  /*
   * Set up receive buffer descriptors
   */
  for (i = 0 ; i < sc->rxBdCount ; i++) {
    (sc->rxBdBase + i)->status = 0;
  }

  /*
   * Set up transmit buffer descriptors
   */
  for (i = 0 ; i < sc->txBdCount ; i++) {
    (sc->txBdBase + i)->status = 0;
    sc->txMbuf[i] = NULL;
  }
  sc->txBdHead = sc->txBdTail = 0;
  sc->txBdActiveCount = 0;
  
  /*
   * Set up interrupts
   */
  status = rtems_interrupt_catch (m823_scc2_interrupt_handler,
				  PPC_IRQ_CPM_SCC2,
				  &old_handler);
  if (status != RTEMS_SUCCESSFUL) {
    rtems_panic ("Can't attach M823 SCC2 interrupt handler: %s\n",
		 rtems_status_text (status));
  }
  m823.scc_reg.sccm = 0;     /* No interrupts unmasked till necessary */
  m823.cpmic.cimr |= (1UL << 29);       /* Enable SCC2 interrupt */
  
    /*
     * Write CICR to Configure SCC1 Interrupt Priority
     * Settings:
     *     SCC Priorities
     *     IRL0-IRL2 (Interrupt Request Level) = (setup parameter)
     *     HP0-HP4 (Highest Priority) = Original Priority
     *     IEN = Enable CPM Interrupts
     */
   /* Josu: Interrupt number 4 */

    m823.cpmic.cicr = 0x931f80 | (4 << 13);
    m823.siu.simask |= (1 << 22);

    /*
     * Initialize GSMR High 32-Bits
     * Settings:  Normal Mode
     */

   m823.scc_reg.gsmr_h = 0;

    /*
     * Initialize GSMR Low 32-Bits, but do not Enable Transmit/Receive
     * Settings:
     *     TCI = Invert
     *     TPL =  48 bits
     *     TPP = Repeating 10's
     *     MODE = Ethernet
     */
 
    m823.scc_reg.gsmr_l = (1<<28) | (4<<21) | (1<<19) | (0xc);

    /*
     * Initialize the DSR -- see section 13.14.4 (pg. 513) v0.4
     */

    m823.scc_reg.dsr = 0xd555;

    /*
     * Initialize the PSMR
     * Settings:
     *      CRC = 32-Bit CCITT
     *      NIB = Begin searching for SFD 22 bits after RENA
     *      PRO = (setup parameter)
     */
   m823.scc_reg.psmr = (2<<10) | (5<<1) | (sc->acceptBroadcast ? 0 : 0x100);
    /*
     * Configure Ethernet TENA=PB18 Signal
     */

    m823.pb.pbpar |= (0x2000);
    m823.pb.pbdir |= (0x2000);

    /*
     * Set the ENT/ENR bits in the GSMR Low -- Enable Transmit/Receive
     */

    m823.scc_reg.gsmr_l |= (1 << 5) | (1 << 4);

    /*
     *  BCSR1 Settings on MPC823FADS:
     *  Assert ETHEN* (=0)
     */
    bcsr[1] &= ~(1<<(31-2));
    /*
     * BCSR4 Settings on MPC823FADS:
     * Deassert ETHLOOP   (=0)
     * Deassert TPFULDL*  (=1)
     * Deassert TPSQEL*   (=1)
     *
     * Note:  The following two settings
     * are necessary for external transmission,
     * only. Since this driver demonstrates external
     * reception, these settings have no effect here,
     * but are provided for instructional purposes.
     *
     * Assert   MODEM_EN   (=0)
     * Deassert DATA_VOICE (=0)
     *
     */

    bcsr[4] &= ~(1<<(31-0));                   /* ETHLOOP */
    bcsr[4] |=  (1<<(31-1)) | (1<<(31-2));     /* TPFULDL, TPSQEL */
    bcsr[4] &= ~((1<<(31-11)) | (1<<(31-12))); /* MODEM_EN, DATA_VOICE */
}


/*
 * Soak up buffer descriptors that have been sent
 * Note that a buffer descriptor can't be retired as soon as it becomes
 * ready.  The MC68360 Errata (May 96) says that, "If an Ethernet frame is
 *  made up of multiple buffers, the user should not reuse the first buffer
 * descriptor until the last buffer descriptor of the frame has had its
 * ready bit cleared by the CPM".
 */
static void
m823Enet_retire_tx_bd (struct m823_enet_struct *sc)
{
  rtems_unsigned16 status;
  int i;
  int nRetired;
  struct mbuf *m, *n;
  
  i = sc->txBdTail;
  nRetired = 0;
  while ((sc->txBdActiveCount != 0)
         &&  (((status = (sc->txBdBase + i)->status) & M823_BD_READY) == 0)) {
    /*
     * See if anything went wrong
     */
    if (status & (M823_BD_DEFER |
                  M823_BD_HEARTBEAT |
                  M823_BD_LATE_COLLISION |
                  M823_BD_RETRY_LIMIT |
                  M823_BD_UNDERRUN |
                  M823_BD_CARRIER_LOST)) {
      /*
       * Check for errors which stop the transmitter.
       */
      if (status & (M823_BD_LATE_COLLISION |
                    M823_BD_RETRY_LIMIT |
                    M823_BD_UNDERRUN)) {
        if (status & M823_BD_LATE_COLLISION)
          enet_driver[0].txLateCollision++;
        if (status & M823_BD_RETRY_LIMIT)
          enet_driver[0].txRetryLimit++;
        if (status & M823_BD_UNDERRUN)
          enet_driver[0].txUnderrun++;
        
        /*
         * Restart the transmitter
         */
        /* FIXME: this should get executed only if using the SCC */
        M823ExecuteRISC (M823_CR_OP_RESTART_TX | M823_CR_CHAN_SCC2);
      }
      if (status & M823_BD_DEFER)
        enet_driver[0].txDeferred++;
      if (status & M823_BD_HEARTBEAT)
        enet_driver[0].txHeartbeat++;
      if (status & M823_BD_CARRIER_LOST)
        enet_driver[0].txLostCarrier++;
    }
    nRetired++;
    if (status & M823_BD_LAST) {
      /*
       * A full frame has been transmitted.
       * Free all the associated buffer descriptors.
       */
      sc->txBdActiveCount -= nRetired;
      while (nRetired) {
        nRetired--;
        m = sc->txMbuf[sc->txBdTail];
        MFREE (m, n);
        if (++sc->txBdTail == sc->txBdCount)
          sc->txBdTail = 0;
      }
    }
    if (++i == sc->txBdCount)
      i = 0;
  }
}

/*
 * reader task
 */
static void
scc_rxDaemon (void *arg)
{
  struct m823_enet_struct *sc = (struct m823_enet_struct *)arg;
  struct ifnet *ifp = &sc->arpcom.ac_if;
  struct mbuf *m;
  rtems_unsigned16 status;
  m823BufferDescriptor_t *rxBd;
  int rxBdIndex;
  
  /*
   * Allocate space for incoming packets and start reception
   */
  for (rxBdIndex = 0 ; ;) {
    rxBd = sc->rxBdBase + rxBdIndex;
    MGETHDR (m, M_WAIT, MT_DATA);
    MCLGET (m, M_WAIT);
    m->m_pkthdr.rcvif = ifp;
    sc->rxMbuf[rxBdIndex] = m;
    rxBd->buffer = mtod (m, void *);
    rxBd->status = M823_BD_EMPTY | M823_BD_INTERRUPT;
    if (++rxBdIndex == sc->rxBdCount) {
      rxBd->status |= M823_BD_WRAP;
      break;
    }
  }
  
  /*
   * Input packet handling loop
   */
  rxBdIndex = 0;
  for (;;) {
    rxBd = sc->rxBdBase + rxBdIndex;
    
    /*
     * Wait for packet if there's not one ready
     */
    if ((status = rxBd->status) & M823_BD_EMPTY) {
      /*
       * Clear old events
       */
      m823.scc_reg.scce = 0x8;
      
      /*
       * Wait for packet
       * Note that the buffer descriptor is checked
       * *before* the event wait -- this catches the
       * possibility that a packet arrived between the
       * `if' above, and the clearing of the event register.
       */
      while ((status = rxBd->status) & M823_BD_EMPTY) {
        rtems_event_set events;
        
        /*
         * Unmask RXF (Full frame received) event
         */
        m823.scc_reg.sccm |= 0x8;
        
        rtems_bsdnet_event_receive (INTERRUPT_EVENT,
                                    RTEMS_WAIT|RTEMS_EVENT_ANY,
                                    RTEMS_NO_TIMEOUT,
                                    &events);
      }
    }
    
    /*
     * Check that packet is valid
     */
    if ((status & (M823_BD_LAST |
                   M823_BD_FIRST_IN_FRAME |
                   M823_BD_LONG |
                   M823_BD_NONALIGNED |
                   M823_BD_SHORT |
                   M823_BD_CRC_ERROR |
                   M823_BD_OVERRUN |
                   M823_BD_COLLISION)) ==
        (M823_BD_LAST |
         M823_BD_FIRST_IN_FRAME)) {
      /*
       * Pass the packet up the chain.
       * FIXME: Packet filtering hook could be done here.
       */
      struct ether_header *eh;
      
      m = sc->rxMbuf[rxBdIndex];
      m->m_len = m->m_pkthdr.len = rxBd->length -
        sizeof(rtems_unsigned32) -
        sizeof(struct ether_header);
      eh = mtod (m, struct ether_header *);
      m->m_data += sizeof(struct ether_header);
      ether_input (ifp, eh, m);
      
      /*
       * Allocate a new mbuf
       */
      MGETHDR (m, M_WAIT, MT_DATA);
      MCLGET (m, M_WAIT);
      m->m_pkthdr.rcvif = ifp;
      sc->rxMbuf[rxBdIndex] = m;
      rxBd->buffer = mtod (m, void *);
    }
    else {
      /*
       * Something went wrong with the reception
       */
      if (!(status & M823_BD_LAST))
        sc->rxNotLast++;
      if (!(status & M823_BD_FIRST_IN_FRAME))
        sc->rxNotFirst++;
      if (status & M823_BD_LONG)
        sc->rxGiant++;
      if (status & M823_BD_NONALIGNED)
        sc->rxNonOctet++;
      if (status & M823_BD_SHORT)
        sc->rxRunt++;
      if (status & M823_BD_CRC_ERROR)
        sc->rxBadCRC++;
      if (status & M823_BD_OVERRUN)
        sc->rxOverrun++;
      if (status & M823_BD_COLLISION)
        sc->rxCollision++;
    }
    
    /*
     * Reenable the buffer descriptor
     */
    rxBd->status = (status & (M823_BD_WRAP | M823_BD_INTERRUPT)) |
                    M823_BD_EMPTY;
    
    /*
     * Move to next buffer descriptor
     */
    if (++rxBdIndex == sc->rxBdCount)
      rxBdIndex = 0;
  }
}

static void
scc_sendpacket (struct ifnet *ifp, struct mbuf *m)
{
  struct m823_enet_struct *sc = ifp->if_softc;
  volatile m823BufferDescriptor_t *firstTxBd, *txBd;
  struct mbuf *l = NULL;
  rtems_unsigned16 status;
  int nAdded;
  
  /*
   * Free up buffer descriptors
   */
  m823Enet_retire_tx_bd (sc);
  
  /*
   * Set up the transmit buffer descriptors.
   * No need to pad out short packets since the
   * hardware takes care of that automatically.
   * No need to copy the packet to a contiguous buffer
   * since the hardware is capable of scatter/gather DMA.
   */
  nAdded = 0;
  txBd = firstTxBd = sc->txBdBase + sc->txBdHead;
  for (;;) {
    /*
     * Wait for buffer descriptor to become available.
     */
    if ((sc->txBdActiveCount + nAdded) == sc->txBdCount) {
      /*
       * Clear old events
       */
      m823.scc_reg.scce = 0x12;
      
      /*
       * Wait for buffer descriptor to become available.
       * Note that the buffer descriptors are checked
       * *before* * entering the wait loop -- this catches
       * the possibility that a buffer descriptor became
       * available between the `if' above, and the clearing
       * of the event register.
       * This is to catch the case where the transmitter
       * stops in the middle of a frame -- and only the
       * last buffer descriptor in a frame can generate
       * an interrupt.
       */
      m823Enet_retire_tx_bd (sc);
      while ((sc->txBdActiveCount + nAdded) == sc->txBdCount) {
        rtems_event_set events;
        
                                /*
                                 * Unmask TXB (buffer transmitted) and
                                 * TXE (transmitter error) events.
                                 */
        m823.scc_reg.sccm |= 0x12;
        rtems_bsdnet_event_receive (INTERRUPT_EVENT,
                                    RTEMS_WAIT|RTEMS_EVENT_ANY,
                                    RTEMS_NO_TIMEOUT,
                                    &events);
        m823Enet_retire_tx_bd (sc);
      }
    }
    
    /*
     * Don't set the READY flag till the
     * whole packet has been readied.
     */
    status = nAdded ? M823_BD_READY : 0;
    
    /*
     *  FIXME: Why not deal with empty mbufs at at higher level?
     * The IP fragmentation routine in ip_output
     * can produce packet fragments with zero length.
     * I think that ip_output should be changed to get
     * rid of these zero-length mbufs, but for now,
     * I'll deal with them here.
     */
    if (m->m_len) {
      /*
       * Fill in the buffer descriptor
       */
      txBd->buffer = mtod (m, void *);
      txBd->length = m->m_len;
      sc->txMbuf[sc->txBdHead] = m;
      nAdded++;
      if (++sc->txBdHead == sc->txBdCount) {
        status |= M823_BD_WRAP;
        sc->txBdHead = 0;
      }
      l = m;
      m = m->m_next;
    }
    else {
      /*
       * Just toss empty mbufs
       */
      struct mbuf *n;
      MFREE (m, n);
      m = n;
      if (l != NULL)
        l->m_next = m;
    }
    
    /*
     * Set the transmit buffer status.
     * Break out of the loop if this mbuf is the last in the frame.
     */
    if (m == NULL) {
      if (nAdded) {
        status |= M823_BD_PAD | M823_BD_LAST | M823_BD_TX_CRC | M823_BD_INTERRUPT;
        txBd->status = status;
        firstTxBd->status |= M823_BD_READY;
        sc->txBdActiveCount += nAdded;
      }
      break;
    }
    txBd->status = status;
    txBd = sc->txBdBase + sc->txBdHead;
  }
}

/*
 * Driver transmit daemon
 */
void
scc_txDaemon (void *arg)
{
  struct m823_enet_struct *sc = (struct m823_enet_struct *)arg;
  struct ifnet *ifp = &sc->arpcom.ac_if;
  struct mbuf *m;
  rtems_event_set events;
  
  for (;;) {
    /*
     * Wait for packet
     */
    rtems_bsdnet_event_receive (START_TRANSMIT_EVENT, RTEMS_EVENT_ANY | RTEMS_WAIT, RTEMS_NO_TIMEOUT, &events);
    
    /*
     * Send packets till queue is empty
     */
    for (;;) {
      /*
       * Get the next mbuf chain to transmit.
       */
      IF_DEQUEUE(&ifp->if_snd, m);
      if (!m)
        break;
      scc_sendpacket (ifp, m);
    }
    ifp->if_flags &= ~IFF_OACTIVE;
  }
}

/*
 * Send packet (caller provides header).
 */
static void
m823_enet_start (struct ifnet *ifp)
{
  struct m823_enet_struct *sc = ifp->if_softc;
  
  rtems_event_send (sc->txDaemonTid, START_TRANSMIT_EVENT);
  ifp->if_flags |= IFF_OACTIVE;
}

/*
 * Initialize and start the device
 */
static void
scc_init (void *arg)
{
  struct m823_enet_struct *sc = arg;
  struct ifnet *ifp = &sc->arpcom.ac_if;
  
  if (sc->txDaemonTid == 0) {
    
    /*
     * Set up SCC hardware
     */
    m823_scc_initialize_hardware (sc);
    
    /*
     * Start driver tasks
     */
    sc->txDaemonTid = rtems_bsdnet_newproc ("SCtx", 4096, scc_txDaemon, sc);
    sc->rxDaemonTid = rtems_bsdnet_newproc ("SCrx", 4096, scc_rxDaemon, sc);
    
  }
  
  /*
   * Set flags appropriately
   */
  if (ifp->if_flags & IFF_PROMISC)
    m823.scc_reg.psmr |= 0x200;
  else
    m823.scc_reg.psmr &= ~0x200;
  
  /*
   * Tell the world that we're running.
   */
  ifp->if_flags |= IFF_RUNNING;
  
  /*
   * Enable receiver and transmitter
   */
  m823.scc_reg.gsmr_l |= 0x30;
}


/*
 * Stop the device
 */
static void
scc_stop (struct m823_enet_struct *sc)
{
  struct ifnet *ifp = &sc->arpcom.ac_if;
  
  ifp->if_flags &= ~IFF_RUNNING;
  
  /*
   * Shut down receiver and transmitter
   */
  m823.scc_reg.gsmr_l &= ~0x30;
}

/*
 * Show interface statistics
 */
static void
enet_stats (struct m823_enet_struct *sc)
{
  printf ("      Rx Interrupts:%-8lu", sc->rxInterrupts);
  printf ("       Not First:%-8lu", sc->rxNotFirst);
  printf ("        Not Last:%-8lu\n", sc->rxNotLast);
  printf ("              Giant:%-8lu", sc->rxGiant);
  printf ("            Runt:%-8lu", sc->rxRunt);
  printf ("       Non-octet:%-8lu\n", sc->rxNonOctet);
  printf ("            Bad CRC:%-8lu", sc->rxBadCRC);
  printf ("         Overrun:%-8lu", sc->rxOverrun);
  printf ("       Collision:%-8lu\n", sc->rxCollision);
  printf ("          Discarded:%-8lu\n", (unsigned long)m823.pram.scc2_spi.e.disfc);
  
  printf ("      Tx Interrupts:%-8lu", sc->txInterrupts);
  printf ("        Deferred:%-8lu", sc->txDeferred);
  printf (" Missed Hearbeat:%-8lu\n", sc->txHeartbeat);
  printf ("         No Carrier:%-8lu", sc->txLostCarrier);
  printf ("Retransmit Limit:%-8lu", sc->txRetryLimit);
  printf ("  Late Collision:%-8lu\n", sc->txLateCollision);
  printf ("           Underrun:%-8lu", sc->txUnderrun);
  printf (" Raw output wait:%-8lu\n", sc->txRawWait);
}

/*
 * Driver ioctl handler
 */
static int
scc_ioctl (struct ifnet *ifp, int command, caddr_t data)
{
  struct m823_enet_struct *sc = ifp->if_softc;
  int error = 0;
  
  switch (command) {
  case SIOCGIFADDR:
  case SIOCSIFADDR:
    ether_ioctl (ifp, command, data);
    break;
    
  case SIOCSIFFLAGS:
    switch (ifp->if_flags & (IFF_UP | IFF_RUNNING)) {
    case IFF_RUNNING:
      scc_stop (sc);
      break;
      
    case IFF_UP:
      scc_init (sc);
      break;
      
    case IFF_UP | IFF_RUNNING:
      scc_stop (sc);
      scc_init (sc);
      break;
      
    default:
      break;
    }
    break;
    
  case SIO_RTEMS_SHOW_STATS:
    enet_stats (sc);
    break;
    
    /*
     * FIXME: All sorts of multicast commands need to be added here!
     */
  default:
    error = EINVAL;
    break;
  }
  return error;
}

/*
 * Attach an SCC driver to the system
 */
int
rtems_scc2_driver_attach (struct rtems_bsdnet_ifconfig *config)
{
  struct m823_enet_struct *sc;
  struct ifnet *ifp;
  int mtu;
  int i;
  
  /*
   * Find a free driver
   */
  for (i = 0 ; i < NIFACES ; i++) {
    sc = &enet_driver[i];
    ifp = &sc->arpcom.ac_if;
    if (ifp->if_softc == NULL)
      break;
  }
  if (i >= NIFACES) {
    printf ("Too many SCC drivers.\n");
    return 0;
  }
  
  /*
   * Process options
   */
  if (config->hardware_address) {
    memcpy (sc->arpcom.ac_enaddr, config->hardware_address, ETHER_ADDR_LEN);
  }
  else {
    sc->arpcom.ac_enaddr[0] = 0x44;
    sc->arpcom.ac_enaddr[1] = 0x22;
    sc->arpcom.ac_enaddr[2] = 0x33;
    sc->arpcom.ac_enaddr[3] = 0x33;
    sc->arpcom.ac_enaddr[4] = 0x22;
    sc->arpcom.ac_enaddr[5] = 0x44;
  }
  if (config->mtu)
    mtu = config->mtu;
  else
    mtu = ETHERMTU;
  if (config->rbuf_count)
    sc->rxBdCount = config->rbuf_count;
  else
    sc->rxBdCount = RX_BUF_COUNT;
  if (config->xbuf_count)
    sc->txBdCount = config->xbuf_count;
  else
    sc->txBdCount = TX_BUF_COUNT * TX_BD_PER_BUF;
  sc->acceptBroadcast = !config->ignore_broadcast;
  
  /*
   * Set up network interface values
   */
  ifp->if_softc = sc;
  ifp->if_unit = i + 1;
  ifp->if_name = "eth";
  ifp->if_mtu = mtu;
  ifp->if_init = scc_init;
  ifp->if_ioctl = scc_ioctl;
  ifp->if_start = m823_enet_start;
  ifp->if_output = ether_output;
  ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX;
  if (ifp->if_snd.ifq_maxlen == 0)
    ifp->if_snd.ifq_maxlen = ifqmaxlen;
  
  /*
   * Attach the interface
   */
  if_attach (ifp);
  ether_ifattach (ifp);
  return 1;
};

int
rtems_enet_driver_attach (struct rtems_bsdnet_ifconfig *config)
{
    return rtems_scc2_driver_attach(config);
}
