#define I_WANT_TERMIOS 
/*
 *  BSP specific Serial I/O Functions for the FADS823 BSP
 *
 * Modified for MPC823 by Josu Onandia <jonandia@fagorautomation.es>
 * Derived from the FADS860 BSP (author unknown to me)
 * The changes are mostly substituing the '860' by '823'
 *
 * This file contains the BSP specific functions for 
 * performing serial I/O. These are the functions
 * RTEMS uses (the 2 listed in the device driver 
 * structure)
 *
 * The SMCs are assigned as follows
 *
 *   Channel     Device      Minor   Termios
 *    SMC1      /dev/console   0       yes
 *    SMC2      /dev/tty       1       no
 *
 * Author: Jay Monkman (jmonkman@frasca.com) 
 * Copyright (C) 1998 by Frasca International, Inc.
 *
 *  $Id: console.c,v 1.2 1999/02/18 15:09:56 joel Exp $
 */

#include <rtems/libio.h>
#include <mpc823.h>
#include <mpc823/console.h>
#include <termios.h>

rtems_device_driver console_initialize(rtems_device_major_number major,
                                       rtems_device_minor_number minor,
                                       void *arg)
{
  rtems_status_code status;
  rtems_isr_entry old_handler;
  rtems_status_code sc;
  
#ifdef I_WANT_TERMIOS
  /*
   * Set up TERMIOS (for /dev/console)
   */
  rtems_termios_initialize();
#endif

  /*
   * Set up Buffer Descriptors
   */
  m823_console_initialize();

  /*
   * Do device-specific initialization
   */
  m823_smc_initialize(1);  /* /dev/console */
  m823_smc_initialize(2);  /* /dev/tty     */
  
  sc = rtems_interrupt_catch (m823_smc1_console_interrupt_handler,
                              PPC_IRQ_CPM_SMC1,
                              &old_handler);
  sc = rtems_interrupt_catch (m823_smc2_console_interrupt_handler,
                              PPC_IRQ_CPM_SMC2,
                              &old_handler);

  /*
   * Register the devices
   */
  status = rtems_io_register_name ("/dev/console", major, SMC1_MINOR);
  if (status != RTEMS_SUCCESSFUL)
    rtems_fatal_error_occurred (status);
  status = rtems_io_register_name ("/dev/tty", major, SMC2_MINOR);
  if (status != RTEMS_SUCCESSFUL)
    rtems_fatal_error_occurred (status);
  return RTEMS_SUCCESSFUL;
}
 
rtems_device_driver console_open(rtems_device_major_number major,
                                 rtems_device_minor_number minor,
                                 void *arg)
{
#ifdef I_WANT_TERMIOS
  static const rtems_termios_callbacks smcPollCallbacks = {
    NULL,                       /* firstOpen */
    NULL,                       /* lastClose */
    m823_char_poll_read,        /* pollRead */
    m823_char_poll_write,       /* write */
    m823_smc_set_attributes,    /* setAttributes */
    NULL,                       /* stopRemoteTx */
    NULL,                       /* startRemoteTx */
    0                           /* outputUsesInterrupts */
  };
#endif /* I_WANT_TERMIOS */

  switch (minor) {
  case 0:
#ifndef I_WANT_TERMIOS
    m823.smc1.smcm = 1;           /* Enable SMC1 RX interrupts */
    m823.cimr |= 1UL <<  4;       /* Enable SMC1 interrupts */
#else
    m823.smc_reg[0].smcm = 0;           /* Disable SMC1 RX interrupts */
    m823.cpmic.cimr &= ~(1UL <<  4);    /* Disable SMC1 interrupts */
#endif /* I_WANT_TERMIOS */
    break;
  case 1:
    m823.smc_reg[1].smcm = 1;           /* Enable SMC2 RX interrupts */
    m823.cpmic.cimr |= 1UL <<  3;       /* Enable SMC2 interrupts */
    break;
  default:
    rtems_panic ("CONSOLE: bad minor number");
  }

#ifdef I_WANT_TERMIOS
  if (minor == SMC1_MINOR) {
    return rtems_termios_open (major, minor, arg, &smcPollCallbacks);
  }
  else { 
    return RTEMS_SUCCESSFUL;
  }
#else
  return RTEMS_SUCCESSFUL;
#endif
}

rtems_device_driver console_close(rtems_device_major_number major,
                                  rtems_device_minor_number minor,
                                  void *arg)
{
#ifdef I_WANT_TERMIOS
  if (minor == SMC1_MINOR) {
    return rtems_termios_close (arg);
  }
  else {
    return RTEMS_SUCCESSFUL;
  }
#else
  return RTEMS_SUCCESSFUL;
#endif
}

rtems_device_driver console_read(rtems_device_major_number major,
                                 rtems_device_minor_number minor,
                                 void *arg)
{
#ifdef I_WANT_TERMIOS
  if (minor == SMC1_MINOR) {
    return rtems_termios_read(arg);
  }
  else {
    return m823_console_read(major, minor, arg);
  }
#else
  return m823_console_read(major, minor, arg);
#endif
}

rtems_device_driver console_write(rtems_device_major_number major,
                                  rtems_device_minor_number minor,
                                  void *arg)
{
#ifdef I_WANT_TERMIOS
  if (minor == SMC1_MINOR) {
    return rtems_termios_write(arg);
  }
  else {
    return m823_console_write(major, minor, arg);
  }
#else
  return m823_console_write(major, minor, arg);
#endif
}

/*
 * Handle ioctl request.
 * Should set hardware line speed, bits/char, etc.
 */
rtems_device_driver console_control(rtems_device_major_number major,
                                    rtems_device_minor_number minor,
                                    void *arg)
{ 
#ifdef I_WANT_TERMIOS
  if (minor == SMC1_MINOR) {
    return rtems_termios_ioctl (arg);
  }
  else {
    return RTEMS_SUCCESSFUL;
  }
#else
  return RTEMS_SUCCESSFUL;
#endif
}

