program install;

uses
  Windows,
  SysUtils,
  Registry;


const
  MaxFileCount     = 10;
  InstallerName    = 'Cygwin for RTEMS';
  InstallerVersion = '20000203';

  CygwinDir        = '\cygnus\cygwin-b20\H-i586-cygwin32\bin';

type
  TFileList = record
                Files : array[0..MaxFileCount] of string;
                Count : integer;
              end;


procedure PrintUsage;
begin
  writeln('Usage:');
  writeln('  install [-s] install-directory');
  writeln;
  writeln('This program will install ' + InstallerName );
  writeln('to the install-directory from the current directory.');
  writeln;
  writeln('-s  : Specifies that the Cygwin drive mounts be system wide');
  writeln('      rather than specific to the current user.');
  writeln;
  writeln('Press return to continue...');
  readln;
end;

function RunProgToCompletion( ProgramName : string; CommandLine : string ):boolean;
var
  StartupInfo : TStartupInfo;
  ProcessInfo : TProcessInformation;
  Succeeded   : boolean;
begin
  {** Init default startup sequence **}
  FillChar(StartupInfo,sizeof(StartupInfo),0);
  with StartupInfo do
  begin
    cb:=sizeof(StartupInfo);
  end;


  {** Actually fire up the new process **}
  Succeeded := CreateProcess( PChar(ProgramName),
                              PChar(CommandLine),
                              nil,nil, FALSE,
                              NORMAL_PRIORITY_CLASS AND DETACHED_PROCESS,
                              nil, nil,
                              StartupInfo,
                              ProcessInfo );
  if Succeeded then
  begin
    {** wait for the program to finish **}
    WaitForSingleObject( ProcessInfo.hProcess, INFINITE );
    Result:=TRUE; {** say we've worked OK **}
  end
  else
    Result:=FALSE; {** say that it failed somewhere **}
end;


procedure CreateBlankRegistry( InstallPoint : string );
var
  Cygwin32Reg : TRegistry;
  Index       : integer;
begin
  Cygwin32Reg := TRegistry.Create;
  with Cygwin32Reg do
  begin
    { cleanout all that went before }
    RootKey := HKEY_CURRENT_USER;
    DeleteKey( '\Software\Cygnus Solutions\Cygwin\mounts v2\' );

    { route around problems with read-only HKEY_LOCAL_MACHINE hive }
    try
      RootKey := HKEY_LOCAL_MACHINE;
      DeleteKey( '\Software\Cygnus Solutions\Cygwin\mounts v2\' );
    except
    end;

    { see if we should install to HKEY_LOCAL_MACHINE or HKEY_CURRENT_USER }
    for Index := 1 to ParamCount
      if UpperCase(ParamStr(Index)) = '-S' then
        RootKey := HKEY_LOCAL_MACHINE
      else
        RootKey := HKEY_CURRENT_USER

    { set the root }
    OpenKey( '\Software\Cygnus Solutions\Cygwin\mounts v2\/', TRUE );
    WriteInteger( 'flags', $0A );
    WriteString( 'native', InstallPoint );

    { set the /bin directory to be the same as the cygwin\bin }
    OpenKey( '\Software\Cygnus Solutions\Cygwin\mounts v2\/bin', TRUE );
    WriteInteger( 'flags', $0A );
    WriteString( 'native', InstallPoint + CygwinDir );

    Free;
  end;

end; {of CreateBlankRegistry}


procedure CreateSetEnvBat( InstallPoint : string );
var
  BATFile : System.TextFile;
begin
  Assign( BATFile, InstallPoint + '\start-cygwin.bat' );
  try
    rewrite( BATFile );
    writeln( BATFile, '@echo off' );
    writeln( BATFile, 'SET MAKE_MODE=UNIX' );
    writeln( BATFile, 'SET PATH=' + InstallPoint + CygwinDir + ';%PATH%' );
    writeln( BATFile, 'SET CYGWIN=binmode' );
    writeln( BATFile, 'SET HOME=/home' );
    writeln( BATFile, 'bash' );
  finally
    CloseFile(BATFile);
  end;
end; { CreateSetEnvBat }


procedure FindInstallerFiles( Path : string; var FileList : TFileList );
var
  FindErr   : integer;
  SearchRec : TSearchRec;
  FileCount : integer;
begin
  FindErr := FindFirst( Path + '*.tgz', faAnyFile, SearchRec );

  FileCount := 0;
  while (FindErr = 0) AND (FileCount <= MaxFileCount) do
  begin
    { check that we don't have a linux file }
    { TODO: Do we really need this for RTEMS? }
    if Pos( 'LINUX', UpperCase( SearchRec.Name ) ) = 0 then
    begin
      FileList.Files[FileCount] := SearchRec.Name;
      FindErr := FindNext( SearchRec );
      inc( FileCount );
    end
    else
      FindErr := FindNext( SearchRec );
  end;
  FindClose( SearchRec );

  FileList.Count := FileCount;
end;


function ConvertDOSToCygwinPath( const Path : string ):string;
var
  index      : integer;
  CygwinPath : string;
begin
  CygwinPath:=Path;
  for index:=1 to Length(CygwinPath) do
    if CygwinPath[index] = '\' then
      CygwinPath[index] := '/'; { swap \ for / }

  Delete(CygwinPath,2,1); { remove the : from c:/coldfire }
  CygwinPath:='//'+CygwinPath;
  Result:=CygwinPath;
end;


procedure UncompressArchive( const InstallPoint : string; const Archive : string );
begin
  writeln( 'tar -xf' + Archive + ' --use-compress-program=./gzip --directory=' + InstallPoint );
  if RunProgToCompletion( 'tar.exe', 'tar -xf' + Archive + ' --use-compress-program=./gzip --directory=' + InstallPoint ) then
    writeln('Completed OK')
  else
    writeln('Unable to expand archive');
end;

procedure CopyIfNewer( FromFile : string; ToFile : string; const InstallPoint : string );
var
  FullName    : string;
  OldFileDate : TDateTime;
  NewFileDate : TDateTime;
  F           : THandle;
begin
  {TODO: Check file exists in source .... }
  FullName := InstallPoint + '\' + ToFile;
  if FileExists( FullName ) then
  begin
    F := FileOpen( FromFile, fmShareDenyNone );
    OldFileDate:=FileDateToDateTime( FileGetDate( F ) );
    FileClose( F );

    F := FileOpen( FullName, fmShareDenyNone );
    NewFileDate := FileDateToDateTime( FileGetDate( F ) );
    FileClose( F );

    if NewFileDate >= OldFileDate then
      CopyFile( PChar(FromFile), PChar(FullName), FALSE )
    else
      writeln( 'Newer ' + ToFile + ' exists!' );

  end
  else
    CopyFile( PChar(FromFile), PChar(FullName), TRUE );
end;


var
  DOSInstallPoint : string;
  CygInstallPoint : string;
  index           : integer;

  FileList        : TFileList;

begin
  writeln('INSTALL - ' + InstallerName );
  writeln;

  if ParamCount<>1 then
  begin
    PrintUsage;
    Halt;
  end;

  FindInstallerFiles( ExtractFilePath( ParamStr(0) ), FileList );

  DOSInstallPoint := ParamStr(1);

  if FileList.Count < 1 then
  begin
    writeln('Unable to find install files(*.tgz)!');
    writeln('Recreating Registry Entries...');
    CreateBlankRegistry(DOSInstallPoint);
    Halt;
  end;

  writeln('Found the following archives:');
  for index := 0 to FileList.Count - 1 do
    writeln( '  ' + FileList.Files[index] );

  writeln;


  writeln( 'Creating ' + DOSInstallPoint + ' directory...' );
  CreateDir( DOSInstallPoint );
  CreateDir( DOSInstallPoint + '\bin' );

  writeln( 'Creating Registry Entries...' );
  CreateBlankRegistry( DOSInstallPoint );

  writeln( 'Creating start-cygwin.bat...' );
  CreateSetEnvBat( DOSInstallPoint );


  CygInstallPoint := ConvertDOSToCygwinPath( DOSInstallPoint );

  writeln( 'Uncompressing to ' + DOSInstallPoint );
  for index := 0 to FileList.Count - 1 do
  begin
    writeln( 'Installing : ' + FileList.Files[index] );
    UncompressArchive( CygInstallPoint, FileList.Files[index] );
    RenameFile( FileList.Files[index], FileList.Files[index] + '.installed' );
  end;

  writeln( 'Copying install files to ' + DOSInstallPoint + CygwinDir );
  CopyIfNewer( 'tar.exe',     'tar.exe',     DOSInstallPoint + CygwinDir );
  CopyIfNewer( 'gzip.exe',    'gzip.exe',    DOSInstallPoint + CygwinDir );
  CopyIfNewer( 'cygwin1.dll', 'cygwin1.dll', DOSInstallPoint + CygwinDir );

  writeln('Install Complete');
end.
